<?php

namespace App\Repositories\Backend;

use App\Models\Log;
use App\Models\Category;
use App\Models\Auth\User;
use Illuminate\Support\Facades\DB;
use App\Exceptions\GeneralException;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\Storage;
use App\Events\Backend\Categories\CategoryCreated;
use App\Events\Backend\Categories\CategoryDeleted;
use App\Events\Backend\Categories\CategoryUpdated;

class DocumentsRepository extends BaseRepository
{
    /**
     * Associated Repository Model.
     */
    const MODEL = Category::class;

    /**
     * Sortable.
     *
     * @var array
     */
    private $sortable = [
        'id',
        'name',
        'serial_number',
        'url',
        'parent_category_id',
        'created_at',
        'updated_at',
    ];

    /**
     * Retrieve List.
     *
     * @var array
     * @return Collection
     */
    public function retrieveList(array $options = [])
    {
        $perPage = isset($options['per_page']) ? (int) $options['per_page'] : 20;
        $orderBy = isset($options['order_by']) && in_array($options['order_by'], $this->sortable) ? $options['order_by'] : 'created_at';
        $order = isset($options['order']) && in_array($options['order'], ['asc', 'desc']) ? $options['order'] : 'desc';
        $query = $this->query()
            ->orderBy($orderBy, $order);

        if ($perPage == -1) {
            return $query->get();
        }

        return $query->paginate($perPage);
    }

    /**
     * @return mixed
     */
    public function getForDataTable()
    {
        $rootId=Category::whereNull('parent_category_id')->whereNull('url')->value('id');
        $userCategories = auth()->user()->categories->pluck('id')->toArray();

        $filteredCategories = auth()->user()->categories->filter(function ($category) use ($userCategories) {
            return !in_array($category->parent_category_id, $userCategories);
        });

        $filteredCategoryIds = $filteredCategories->pluck('id')->toArray();
        $userDocuments = auth()->user()->documents->pluck('id')->toArray();
        $documents= $this->query()
        ->leftjoin('users', 'users.id', '=', 'doc_categories.created_by')
        ->leftJoin('doc_categories AS parent_category', 'parent_category.id', '=', 'doc_categories.parent_category_id')
        ->whereNotNull('doc_categories.url')
        ->where(function ($query) use ($rootId, $userDocuments) {

                       if(\request()->get('category_id')==null){
                    if(auth()->user()->categories->contains('id', $rootId))
                    {
                        $query->where('doc_categories.parent_category_id', $rootId);
                    }else{
                    if (!empty($userDocuments)) {

                    foreach($userDocuments as $doc){
                        $Document=Category::findorFail($doc);
                        if(!auth()->user()->categories->contains('id', $Document->parent_category_id)){
                            $query->orwhere('doc_categories.id', $doc);
                        }else
                            $query->whereRaw('1 = 0');
                    }
                }

                 }}
                 else{
                    $query->where('doc_categories.parent_category_id',\request()->get('category_id'));
                   }
        })
        ->select([
            'doc_categories.id',
            'doc_categories.name',
            'doc_categories.serial_number',
            'doc_categories.version',
            'doc_categories.url',
            'doc_categories.size',
            'doc_categories.size_s3',
            'doc_categories.expire_date',
            'doc_categories.created_by',
            'doc_categories.created_at',
            'users.id AS user_id',
            'parent_category.name AS category',
            'parent_category.id AS category_id',
        ]);
        $categories=  $this->query()
        ->leftjoin('users', 'users.id', '=', 'doc_categories.created_by')
        ->leftJoin('doc_categories AS parent_category', 'parent_category.id', '=', 'doc_categories.parent_category_id')
        ->whereNull('doc_categories.url')
        ->where(function($query)use ($rootId, $filteredCategoryIds) {

            if(\request()->get('category_id')==null){
                if(auth()->user()->categories->contains('id', $rootId))
                {
                    $query->where('doc_categories.parent_category_id', $rootId);
                }else{
                $query->whereIn('doc_categories.id', $filteredCategoryIds);
                }
             }else {
                $query->where('doc_categories.parent_category_id',\request()->get('category_id'));
            }
        })
        ->select([
            'doc_categories.id',
            'doc_categories.name',
            'doc_categories.serial_number',
            'doc_categories.version',
            'doc_categories.url',
            'doc_categories.size',
            'doc_categories.size_s3',
            'doc_categories.expire_date',
            'doc_categories.created_by',
            'doc_categories.created_at',
            'users.id AS user_id',
            'parent_category.name AS category',
            'parent_category.id AS category_id',
        ]);
        $all = $categories;
        /** @var \App\Models\Auth\User */
        $user = \auth()->user();
        if(!empty($userDocuments)|| $user->isAdministrator()){
            $all = $documents->union($categories);
        }
        $all = $all->get();

        return $all;
    }
    public static function generateSerial() {
        $serial_number = Category::whereNotNull('url')->orderBy('created_at','DESC')->first()->serial_number??0;
        do{
            $serial_number++;
            $serial_number = \str_pad($serial_number,6,'0',\STR_PAD_LEFT);
        }while(Category::whereNotNull('url')->where('serial_number',$serial_number)->exists());
        return $serial_number;
    }
    /**
     * @param array $input
     *
     * @throws \App\Exceptions\GeneralException
     *
     * @return bool
     */
    public function create(array $input)
    {


        $filesystem = Storage::disk('local');
        $documents = [];
        $filenames = explode(",",  $input['name'][0]);
        foreach ($input['file'] as $index => $file) {
            $serial_number= self::generateSerial();
            $name = $filenames[$index].'('.$serial_number.')'.'.'.$file->getClientOriginalExtension();
            $filesystem->put('uploads/documents/'.$name, $file->get());
            $documentInput = [
                'serial_number' =>$serial_number,
                'created_by' => auth()->user()->id,
                'parent_category_id' => $input['categories'] ?? '',
                'url' => 'documents/'.$name,
                'name'=>$filenames[$index],
                'size'=>$file->getSize(),
                'size_s3'=>$file->getSize(),
                'expire_date'=> $input['expire_date'] ?? '',
            ];
            if ($Document = Category::create($documentInput)) {
               Category:: updateParentCategoriesSizes($Document->parent_category_id,$Document->size,true);
                $user =User::find(auth()->user()->id);
                $permissions=Category::FULLPERMISSION;
                if( $user->isGuest())
                {
                    $permissions=Category::FULLPERMISSION - Category::SHARE;
                }
                if (!$user->documents->contains('id', $Document->id)) {
                    $user->documents()->attach($Document->id, ['permissions' => $permissions]);
                }
                event(new CategoryCreated($Document));

                $log_data = [
                    'model' => 'Document',
                    'action' => 'Create', // Create, Update, Delete
                    'user_id' => auth()->user()->id,
                    'record_id' => $Document->id,
                    'label' => $Document->name,
                    'description' => '',
                ];
                Log::create($log_data);

                $documents[] = $Document;
            }
        }
        return $documents;
        throw new GeneralException(__('exceptions.backend.documents.create_error'));
    }

    /**
     * @param \App\Models\Category $Category
     * @param array $input
     */
    public function update(Category $Category, array $input)
    {
        return DB::transaction(function () use ($Category, $input) {
        $old_category = clone $Category;
        $input['updated_by'] = auth()->user()->id;
        $logs = [];
        if (isset($input['file']) && $input['file'] != null) {
            $filesystem = Storage::disk('local');
            $name = $input['name'] .'('.$Category->serial_number.')'. '.' . $input['file']->getClientOriginalExtension();
            $filesystem->put('uploads/documents/' . $name, $input['file']->get());
            $input['url'] = 'documents/' . $name;
            $logs[] = [
                'description' => "The New File Upload : ".$input['name'] ,
                'model' => 'Document',
                'action' => 'Update',
                'user_id' => auth()->user()->id,
                'record_id' => $Category->id,
                'label' => $Category->name,
            ];
        }
        if (isset($input['name']) && !isset($input['file'])) {
            $currentFilePath = 'uploads/' . $Category->url;
            $newFilePath = '';
            if ($Category->name != $input['name']) {
            if (Storage::disk('local')->exists($currentFilePath)) {
                $pathInfo = pathinfo($currentFilePath);
                $newFilePath = $pathInfo['dirname'] . '/' . $input['name'] .'('.$Category->serial_number.')'. '.' . $pathInfo['extension'];

                Storage::disk('local')->move($currentFilePath, $newFilePath);
            }
            $input['url'] = str_replace('uploads/', '', $newFilePath);

                $logs[] = [
                    'description' => "The File Name Change From: " . $Category->name . " To: " . $input['name'],
                    'model' => 'Document',
                    'action' => 'Update',
                    'user_id' => auth()->user()->id,
                    'record_id' => $Category->id,
                    'label' => $Category->name,
                ];
            }

        }

        Category:: updateParentCategoriesSizes($Category->parent_category_id,$Category->size,false);//decrease size old document

        if ($Category->update($input)) {
            Category:: updateParentCategoriesSizes($Category->parent_category_id,$Category->size,true);//increase size new document
            event(new CategoryUpdated($Category));

            foreach ($logs as $log) {
                Log::create($log);
            }
            $this->UpdateVersion($old_category,$Category);
            return $Category->fresh();
        }
        throw new GeneralException(__('exceptions.backend.documents.update_error'));
    });
    }

    /**
     * @param \App\Models\Category $Category
     *
     * @throws GeneralException
     *
     * @return bool
     */
    public static function delete(Category $Category)
    {
        if (Category::where('url', $Category->url)->count() == 1   ) {
           Storage::disk('local')->delete('uploads/'.$Category->url);
        }
        $log_data = [
            'model'   => 'Document',
            'action'    => 'Delete', //Create,Update,Delete
            'user_id' => auth()->user()->id,
            'record_id' => $Category->id,
            'label'  => $Category->name,
            'description' => '',
      ];
         Log::create($log_data);
         $parent_category_id=$Category->parent_category_id;
         $size =$Category->size;
        if ($Category->delete()) {
            Category:: updateParentCategoriesSizes($parent_category_id, $size,false);
            return true;
        }

        throw new GeneralException(__('exceptions.backend.documents.delete_error'));
    }

    /**
     * @param $old_version
     * @param $Category
     */
    public function UpdateVersion($old_version,$Category)
    {
        return DB::transaction(function () use ($old_version, $Category) {
        $Category->version = (int)$old_version->version + 1;
        $Category->name    = $old_version->name;
        if ($Category->save()) {
            $log_data = [
                'model' => 'Document',
                'action' => 'Update', // Create, Update, Delete
                'user_id' => auth()->user()->id,
                'record_id' => $Category->id,
                'label' => $old_version->name,
                'description' => 'update version to ' . $Category->version,
            ];
            Log::create($log_data);
            //?? $serial_number = self::generateSerial();
            $oldVersion = [
                'serial_number' => $Category->serial_number,
                'version' => $old_version->version,
                'created_by' => auth()->user()->id,
                'parent_category_id' => $Category->id,
                'url' => $old_version->url,
                'name' => $Category->name,
                'size' => $Category->size ?: null,
                'size_s3' => $Category->size_s3 ?: null,
                'expire_date' => $Category->expire_date ?: null,
            ];
            if ($Document = Category::create($oldVersion)) {
                $user = User::find(auth()->user()->id);
                $permissions = Category::FULLPERMISSION;
                if ($user->isGuest()) {
                    $permissions = Category::FULLPERMISSION - Category::SHARE;
                }
                if (!$user->documents->contains('id', $Document->id)) {
                    $user->documents()->attach($Document->id, ['permissions' => $permissions]);
                }
                event(new CategoryCreated($Document));

                $log_data = [
                    'model' => 'Document',
                    'action' => 'Create', // Create, Update, Delete
                    'user_id' => auth()->user()->id,
                    'record_id' => $Document->id,
                    'label' => $Document->name,
                    'description' => 'save old version for ' . $Category->id,
                ];
                Log::create($log_data);
                return true;
            }
        }
        throw new GeneralException(__('exceptions.backend.documents.update_error'));
    });
    }

    /**
     * @return mixed
     * @throws \Psr\Container\ContainerExceptionInterface
     * @throws \Psr\Container\NotFoundExceptionInterface
     */
    public function getForDataTableVersions()
    {
        $categories = $this->query()
            ->leftjoin('users', 'users.id', '=', 'doc_categories.created_by')
            ->leftJoin('doc_categories AS parent_category', 'parent_category.id', '=', 'doc_categories.parent_category_id')
            ->select([
                'doc_categories.id',
                'doc_categories.name',
                'doc_categories.serial_number',
                'doc_categories.version',
                'doc_categories.url',
                'doc_categories.size',
                'doc_categories.size_s3',
                'doc_categories.expire_date',
                'doc_categories.created_by',
                'doc_categories.created_at',
                'users.id AS user_id',
                'parent_category.name AS category',
                'parent_category.id AS category_id',
            ])->where('doc_categories.parent_category_id', \request()->get('category_id'))
            ->orderBy('doc_categories.version', 'desc');
        $all = $categories;
        $all = $all->get();
        return $all;
    }
}
