<?php

namespace App\Http\Controllers\Backend\Documents;

use Carbon\Carbon;
use App\Models\Log;
use App\Models\Category;
use App\Models\Auth\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Responses\ViewResponse;
use Illuminate\Support\Facades\View;
use App\Enums\IsReadNotificationEnum;
use App\Http\Responses\RedirectResponse;
use Illuminate\Support\Facades\Response;
use Yajra\DataTables\Facades\DataTables;
use App\Repositories\Backend\DocumentsRepository;
use App\Http\Responses\Backend\Document\EditResponse;
use App\Http\Requests\Backend\Documents\StoreDocumentsRequest;
use App\Http\Requests\Backend\Documents\UpdateDocumentsRequest;


class DocumentsController extends Controller
{
    /**
     * @var \App\Repositories\Backend\DocumentsRepository
     */
    protected $repository;

    /**
     * @param \App\Repositories\Backend\DocumentsRepository $Document
     */
    public function __construct(DocumentsRepository $repository)
    {
        $this->repository = $repository;
        View::share('js', ['documents','categories']);
    }

    /**
     * @param \Illuminate\Http\Request  $request
     *
     * @return ViewResponse
     */
    public function index(Request $request)
    {
        return new ViewResponse('backend.documents.index');
    }

    /**
     * @param \Illuminate\Http\Request  $request
     *
     * @return ViewResponse
     */
    public function create(Request $request)
    {
        $Categories = [];
        $categories = Category::whereNull('parent_category_id')->get();
        foreach ($categories as $category) {
            $Categories[$category->id] = $category->name;
        }
        $category_id = $request->get('categoryid');
        $category=Category::find($category_id );
        return new ViewResponse('backend.documents.create', ['Categories' => $Categories,'category'=>$category]);
    }

    /**
     * @param App\Http\Requests\Backend\Documents\StoreDocumentsRequest  $request
     *
     * @return \App\Http\Responses\RedirectResponse
     */
    public function store(StoreDocumentsRequest $request)
    {
        $this->repository->create($request->except('_token'));
        if($request->get('tree')){
            return new RedirectResponse(route('admin.categories.index', ['categoryid' => $request->get('categories')]), ['flash_success' => __('alerts.backend.documents.created')]);

        }
        return new RedirectResponse(route('admin.documents.index'), ['flash_success' => __('alerts.backend.documents.created')]);
    }

    /**
     * @param $categoryid
     * @param \Illuminate\Http\Request  $request
     *
     * @return ViewResponse
     */
    public function edit( $categoryid, Request $request)
    {
        $category = Category::find( $categoryid);
        $Category = Category::whereNull('url')->get();
        $Categories= Category:: getItems($Category,'name');
        if($category){

        \App\Models\Notification::where('url',route('admin.documents.edit',$categoryid))
        ->where('notifiable_id', '=', Auth()->user()->id)
        ->where('is_read',false)
        ->update(['is_read' => IsReadNotificationEnum::yes,'read_at'    => \date('Y-m-d H:i:s')]);

        }
        return new EditResponse($category, $Categories);
    }

    /**
     * @param $categoryid
     * @param App\Http\Requests\Backend\Documents\UpdateDocumentsRequest  $request
     *
     * @return \App\Http\Responses\RedirectResponse
     */
    public function update($categoryid, UpdateDocumentsRequest $request)
    {
        $category = Category::find($categoryid);
        $this->repository->update($category, $request->except(['_token', '_method']));
       /* if(Auth()->user()->guest==1){
            return new RedirectResponse(route('admin.categories.index'), ['flash_success' => __('alerts.backend.documents.updated')]);
        }
        */
        if($request->get('tree')){
            return new RedirectResponse(route('admin.categories.index', ['categoryid' => $request->get('categories')]), ['flash_success' => __('alerts.backend.documents.updated')]);
        }
        return new RedirectResponse(route('admin.categories.index'), ['flash_success' => __('alerts.backend.documents.updated')]);
    }

    /**
     * @param $categoryid
     * @param \Illuminate\Http\Request  $request
     *
     * @return \App\Http\Responses\RedirectResponse
     */
    public function destroy($categoryid, Request $request)
    {

        $category = Category::find($categoryid);

        $this->repository->delete($category);

        return new RedirectResponse(route('admin.categories.index', ['categoryid' => $category->parent_category_id]), ['flash_success' => __('alerts.backend.documents.deleted')]);
    }

    public function getCategories(Request $request)
    {
        if ($request->ajax()) {
            $categoryId = $request->get('category_id');
            $category = Category::find($categoryId);
            $children_categories = $category->childern_categories;
            $results['children_categories'] = $children_categories;
            echo json_encode($results);
            exit;
        }
    }
    public function managementFilter()
    {
        $Allusers = [];
        $users = User::whereNull('deleted_at')->get();
        foreach ($users as $user) {
            if(!$user->isAdministrator())
            $Allusers[$user->id] = $user->name;
        }
        return view('backend.documents.document_filter',['users'=>$Allusers]);
    }
    public function management(Request $request)
    {
        if($request->ajax()){

            $documents = Category::leftJoin('doc_categories AS documents', 'doc_categories.id', '=', 'documents.parent_category_id')
                  ->leftjoin('users', 'users.id', '=', 'documents.created_by')
                  ->whereNotNull('documents.url')->whereNull('documents.deleted_at')
                  ->where(function($query) use($request){
                $category = $request->get('categories');
                $file_name = $request->get('file_name');
                $file_type = $request->get('filetype');
                $user_id = $request->get('user_id');
                $from = $request->get('from');
                $to = $request->get('to');
                if(!empty($file_name)) $query->where('documents.name', 'LIKE', "%{$file_name}%");
                if (!empty($file_type))$query->whereRaw("SUBSTRING_INDEX(SUBSTRING_INDEX(documents.url, '.', -1), '?', 1) = '{$file_type}'");
                if(!empty($category)) $query->where('documents.parent_category_id',$category);
                if(!empty($user_id)) $query->where('documents.created_by',$user_id);
                if(!empty($from)) $query->whereRaw('DATE(documents.created_at) >= \''.$from.'\'');
                if(!empty($to)) $query->whereRaw('DATE(documents.created_at) <= \''.$to.'\'');
            })->select('documents.*', 'documents.name as category','users.name AS user_name');
            return Datatables::of($documents)
            ->escapeColumns(['name','serial_number'])
            ->addColumn('url',function($doc){
                $doc->attachment_url = $doc->url;
                return '<div class="td-attachment position-relative">'.
                            view('backend.documents.includes.document_file',['document'   => $doc])->render().'</div>';
            })
            ->filterColumn('created_by', function ($query, $keyword) {
                $query->whereRaw('users.first_name like ?', ["%{$keyword}%"]);
            })
            ->editColumn('created_at', function ($documents) {
                return Carbon::parse($documents->created_at)->toDateString();
            })
            ->filterColumn('file_type', function ($query, $keyword) {
                $query->whereRaw('documents.url  LIKE \'%'.$keyword.'%\'');
            })
            ->addColumn('file_type', function ($documents) {
                return \substr($documents->url,\strrpos($documents->url,'.') + 1);
            })
            ->addColumn('category', function ($documents) {
                return $documents->category;
            })
            ->addColumn('size', function ($documents) {
                return formatFileSize($documents->size);
            })
            ->addColumn('size_s3', function ($documents) {
                return formatFileSize($documents->size_s3);
            })
            ->editColumn('expire_date', function ($documents) {
                return $documents->expire_date? Carbon::parse($documents->expire_date)->toDateString():'';
            })
            ->addColumn('created_by', function ($category) {
                return $category->user_name;
            })
            ->make(true);
        }

        return view('backend.documents.report');
    }
    public function search(Request $request)
    {
        $categoryid = trim($request->categoryid);
        if (empty($categoryid)) {
            return response()->json([]);
        }
        if(access()->user()->isAdministrator()){
            $documents=Category::WhereNotNull('url')->where('parent_category_id',$categoryid)->get();
        }else{
            $documents = auth()->user()->documents
            ->where('parent_category_id', $categoryid);
        }
        $file_name = $documents->map(function ($results) {

            $data["id"] = $results->name;
            $data["text"] = $results->name.' ('.$results->serial_number.')';
            return $data;
        });

        $file_type = $documents->pluck('url')->map(function ($url) {
            $extension = pathinfo($url, PATHINFO_EXTENSION);
            return [
                'id' => $extension,
                'text' => $extension,
            ];
        })->unique('id');

        $result['file_name'] = $file_name->toArray();
        $result['file_type'] = $file_type->toArray();
        return response()->json($result);

    }
    public function checkUserDocument(Request $request)
    {
        $userId = $request->input('userId');
        $documentid = $request->input('documentid');
        $permissions = -1;
        $user = User::findOrFail($userId);
        if ($user->documents->contains('id', $documentid)) {
            $permissions=User::getPermission($user,$documentid);
        }
        return response()->json($permissions);
    }
    public function giveUserDocument(Request $request)
    {
        $userId = $request->input('userId');
        $categoryId = $request->input('categoryid');
        $documentid = $request->input('documentid');
        $permission = $request->input('permissions');
        $user = User::findOrFail($userId);
        if (!empty($permission)) {
            $permissions = array_sum($permission);
            if (!$user->documents->contains('id', $documentid)) {
                $user->documents()->attach($documentid, ['permissions' => $permissions]);
            }
            $document= Category::find($documentid);
            $des='';
            if ($permissions == Category::EDIT ||  $permissions == (Category::EDIT | Category::DELETE) || $permissions == (Category::SHARE | Category::EDIT )|| $permissions == (Category::SHARE | Category::EDIT | Category::DELETE)) {
                $des.=' , edit';
            }
            if ($permissions == Category::DELETE || $permissions == (Category::EDIT | Category::DELETE )  || $permissions == (Category::SHARE | Category::DELETE )|| $permissions == (Category::SHARE | Category::EDIT | Category::DELETE)) {
                $des.=' , delete';
            }
            if ($permissions == Category::SHARE || $permissions == (Category::SHARE | Category::DELETE )|| $permissions == (Category::SHARE | Category::EDIT )|| $permissions == (Category::SHARE | Category::EDIT | Category::DELETE)) {
                $des.=' , share';
            }
             $log_data = [
                'model' => 'Permission',
                'action' => 'Update', // Create, Update, Delete
                'user_id' =>  auth()->user()->id,
                'record_id' => $documentid,
                'label' => $document->name.' ('.$user->name.')',
                'description' => 'The '.$document->name.' document has these permissions : view '.$des.' document' .' for :('.$user->name.')',
            ];
            Log::create($log_data);
            return response()->json(['success' => true,'message'=>__('alerts.backend.documents.userdocument')]);
        }


        return response()->json(['success' => false]);
    }
    public function documentpermission(Request $request)
    {
        $users= User::all();
        $document_id = $request->get('document_id');
        $usersWithAccess=[];
        foreach($users as $user)
        {
            if ($user->documents->contains('id', $document_id))
            {
                $usersWithAccess[] = $user;

            }
        }
         return new ViewResponse('backend.documents.document_permission',['users' => $usersWithAccess,'document_id'=>$request->get('document_id')]);
    }
    public function deletedocumentuser(Request $request)
    {
        $userId = $request->input('user_id');
        $document_id = $request->input('document_id');
        $category_id = Category::where('id', $document_id)->pluck('parent_category_id');
        $user = User::with('documents')->findOrFail($userId);
        $user->documents()->detach($document_id);
        $documentsCount = $user->documents()
        ->where('parent_category_id', $category_id)
        ->count();
        $childern_categories = Category::where('id', $category_id)->first()->childern_categories()->count();
        if($documentsCount==0 && $childern_categories==0){
            $user->categories()->detach($category_id);
        }
        $document= Category::find($document_id);
        $log_data = [
            'model' => 'Permission',
            'action' => 'Delete', // Create, Update, Delete
            'user_id' =>  auth()->user()->id,
            'record_id' => $document_id,
            'label' => $document->name .' ('.$user->name.')',
            'description' => 'Permissions deleted from :('.$user->name.')',
        ];
        Log::create($log_data);
        return new RedirectResponse(route('admin.documentpermission',['document_id' => $document_id]), ['flash_success' => __('alerts.backend.access.permissions.deletedcategory')]);
    }
    public function editUserDocument(Request $request)
    {
        $userId = $request->input('userId');
        $documentId = $request->input('documentid');
        $permission = $request->input('permissions');
        $user = User::with('documents')->findOrFail($userId);
        if (!empty($permission)) {
            $permissions = array_sum($permission);
            $user->documents()->updateExistingPivot($documentId, ['permissions' => $permissions]);
             $document= Category::find($documentId);
            $des='';
            if ($permissions == Category::EDIT ||  $permissions == (Category::EDIT | Category::DELETE) || $permissions == (Category::SHARE | Category::EDIT )|| $permissions == (Category::SHARE | Category::EDIT | Category::DELETE)) {
                $des.=' , edit';
            }
            if ($permissions == Category::DELETE || $permissions == (Category::EDIT | Category::DELETE )  || $permissions == (Category::SHARE | Category::DELETE )|| $permissions == (Category::SHARE | Category::EDIT | Category::DELETE)) {
                $des.=' , delete';
            }
            if ($permissions == Category::SHARE || $permissions == (Category::SHARE | Category::DELETE )|| $permissions == (Category::SHARE | Category::EDIT )|| $permissions == (Category::SHARE | Category::EDIT | Category::DELETE)) {
                $des.=' , share';
            }
             $log_data = [
                'model' => 'Permission',
                'action' => 'Update', // Create, Update, Delete
                'user_id' =>  auth()->user()->id,
                'record_id' => $documentId,
                'label' => $document->name.' ('.$user->name.')',
                'description' => 'The '.$document->name.' document changed its permissions to : view'.$des.' document'.' for :('.$user->name.')',
            ];
            Log::create($log_data);
            return response()->json(['success' => true,'message'=>__('alerts.backend.categories.usercategoryupdate')]);

        }


        return response()->json(['success' => false]);
    }

    /***
     * @param Request $request
     * @return string
     * @throws \Psr\Container\ContainerExceptionInterface
     * @throws \Psr\Container\NotFoundExceptionInterface
     */
    public function getAllVersion(Request $request){
        $data = $this->repository->getForDataTableVersions();
        $listItems = '';
        foreach ($data as $document) {
            $fileType = substr($document->url, strrpos($document->url, '.') + 1);
            $folderClass = $document->url != null ? '' : 'Folder';
            $permissions=$folderClass=='Folder'?'':User::getPermission(auth()->user(), $document->id);
            $user=User::find($document->user_id);
            $username=$user?$user->full_name: '';
            $userpicture=$user?$user->picture:'' ;
            $path=$document->url;
            $listItems .= '<li class="data-content list" data-permission="'.$permissions.'" data-id="'.$document->id.'" style="display: none;">' .
                '<span class="serial_number">' . $document->serial_number . '</span>' .
                '<span class="icon">' . '<div class="td-attachment ' . $folderClass . '" data-category="' . $document->id . '">' .
                view('backend.documents.includes.document_file', ['document' => $document])->render() .
                '</div>' . '</span>' .
                '<span class="name">' . $document->name . '</span>' .
                '<span class="type">' . ($fileType ?: 'Folder') . '</span>' .
                '<span class="category">' . $document->category . '</span>' .
                '<span class="version">' . ($document->version ?: '0') . '</span>'.
                '<span class="icon"><img src="'. $userpicture. '" class="img-2"></span>'.
                '<span class="owner">' .$username. '</span>' .
                '<input type="hidden" id="documentUrl" value="'.$path.'">'.
                '</li>';
        }
        if($listItems== ''){
            $listItems = '<div class="msg-data">No data available</div>';
        }
        return $listItems;
    }
}
